; FUNCTION imfixrow,image,omask,badmask,ORDER=order

; this file contains:
; FUNCTION imfixrow - repairs ALL flagged pixels in an image (if possible)
; PRO      pixfixrow - repairs ONE flagged pixel in an image (if possible)
;
;  5 Dec 05 changed names, imfixrow (was imbadfix), pixfixrow (was replace_bad)
;  5 Jan 05 tidying up comments
; 10 Dec 04 GS separated old imclean.pro into imclean.pro and imbadfix.pro
; 28 Jul 04 major revisions by Mark Keremedjiev
; 27 Jul 04 added custom mask feature
; 21 May 04 stylistic changes and comments by MJR and GS
; 18 May 04 created by Mike Russell
;
; given a mask of bad pixels and an order mask, repair flagged pixels in 
;  an image

; INPUT
;   image     2-D image
;   omask     the 2-D order mask
;   badmask   the 2-D mask of pixels flagged as bad
;   order     optional parameter to limit repairs to one order
;             encouraged for SL and LL
;   additional parameters passed to pixfixrow
; OUTPUT returns cleaned image
;
; NOTE if a row contains fewer than 3 good pixels in a given order,
;  it is not repaired


;------------------------------------------------------------------------------


PRO pixfixrow,im,badmask,x,y,x_idx,NROW=nrow,LIMIT=limit,VERBOSE=verbose

;  5 Dec 05 renamed pixfixrow from replace_bad
;  2 Aug 05 added limit keyword to prevent normalization to noise
; 27 May 05 do not replace a pixel if the column weights total 0
; 20 Dec 04 getting the bugs out
; 17 Dec 04 major revision by GS to simplify algorithm and make it more robust
; 29 Jul 04 renamed replace_bad from replace_nan
; 21 May 04 stylistic changes and comments by MJR and GS
; 18 May 04 created by MJR

; replaces a bad pixel in an image
; called by imfixrow.pro
; cleans by averaging spatial profiles in rows above and below affected row
; then normalize average profile to affected row and replace bad pixel
; the profiles are fitted to the good data in the affected row
;   by minimizing chi^2
; the number of rows used can be changed by changing numrows

; INPUT
;   im       2D image
;   badmask  2D image containing 0 for good data, 1 for bad data
;   x        x-coordinate of bad pixel to be replaced
;   y        y-coordinate of bad pixel to be replaced
;   x_idx    indices of pixels within order in row y
;   nrow     optional keyword, sets number of rows above and below problem row 
;            to be used to build spatial profile, defaults to 3
;   limit    optional keyword, pixels in row with bad data not used for
;              normalization if flux < limit*total profile, defaults to 0.025
;   verbose  optional keyword to turn on diagnostics
; OUTPUT - returns cleaned image (as INPUT parameter im)

; check keywords

if (keyword_set(verbose) eq 0) then verboseflag=0 else verboseflag=1
if (keyword_set(nrow) eq 0) then nrow=3      ; DEFAULT
if (keyword_set(limit) eq 0) then limit=0.025 ; DEFAULT

; generate the test profile from the problem row

test_prof=im[x_idx,y]
test_mask=badmask[x_idx,y]

; generate the average profile from the rows above and below the problem row

miny=0 
maxy=n_elements(im[0,*])-1
y_idx=y-nrow+findgen(2*nrow+1)
y_idx=y_idx[where(y_idx ge 0 and y_idx le maxy and y_idx ne y)]

xlen=n_elements(x_idx)
avg_prof=fltarr(xlen)
wt_prof =fltarr(xlen)               ; profile of weights for avg_prof
for i=0,xlen-1 do begin             ; build average profile
  data=im[x_idx[i],y_idx]           ; data to sum into avg_profile
  weight=-badmask[x_idx[i],y_idx]+1 ; inverts relevant parts of badmask
  idx=where(finite(data) eq 0)      ; NaNed data
  if (max(idx) gt -1) then begin
    weight[idx]=0 & data[idx]=0     ; zero weight and data where NaNed
  endif
  avg_prof[i]=total(data*weight)    ; /nan in total now redundant
  wt_prof[i]=total(weight)          ; divide by total weight unless zero
  if (wt_prof[i] gt 0.0) then avg_prof[i]=avg_prof[i]/wt_prof[i] $
    else avg_prof[i]=0.0
endfor

; set test_mask to 1 where test_prof < limit*total(avg_prof)
; this prevents using pixels which are just noise to normalize the profiles

limit_idx=where(test_prof lt limit*total(avg_prof))
if (max(limit_idx) gt -1) then test_mask[limit_idx]=1

; if we have enough valid data, then 
; normalize the average profile to the good data in the test profile
; by using least-squares fitting (= sum(avg_prof*test_prof) / sum(avg_prof^2)
; and replace bad datum with datum from averaged profile

norm_prof=fltarr(xlen)
good_idx=where(avg_prof ne 0.0 and test_mask eq 0)

if (x eq 97 and y eq 30) then begin
  print,avg_prof,[total(avg_prof),limit*total(avg_prof)]
  print,test_prof
endif

x_spot=where(x eq x_idx)             ; pixel to replace, in local units

; don't replace pixel if we have no good data or if weights in column total 0

if (max(good_idx) gt -1 and wt_prof[x_spot] gt 0.0) then replace_flag=1 $
  else replace_flag=0

if (replace_flag eq 1) then begin
  knum = total( avg_prof[good_idx] * test_prof[good_idx] )
  kden = total( avg_prof[good_idx]^2 )
  norm_prof = avg_prof*knum/kden ; normalize profile to good data in bad row

; replace the bad pixel - profile and image are indexed differently
; so use where(x eq x_idx) 

  im[x,y] = norm_prof[x_spot] 
  if (verboseflag eq 1) then $
    print,'Replacing x:',x,'  y:',y,' flux:',norm_prof[x_spot]

endif else begin
  if (verboseflag eq 1) then print,'Not replacing x:',x,'  y:',y
endelse

END


;------------------------------------------------------------------------------


FUNCTION imfixrow,image,omask,badmask,ORDER=order,_extra=e

; see comment header at the top of the file

newimage = image
ncol=n_elements(image[*,0])
nrow=n_elements(image[0,*])

; check keywords

if (n_elements(order) eq 1) then ordflag=1 else ordflag=0

; determine orders

if (ordflag eq 0) then begin
  omin = 1
  omax = max(omask)
endif else begin
  omin = order
  omax = order
endelse

for i=omin,omax do begin   ; loop through all orders

  for j=0,nrow-1 do begin     ; check all rows for a given order

;   must have pixels for specified order in current row 

    if (max(where(omask[*,j] eq i) gt -1)) then begin
      idx = where(omask[*,j] eq i)

      pix_idx = where(badmask[idx,j] eq 1)

      if (max(pix_idx) gt -1) then begin ; replace flags one by one

        for k=0,n_elements(pix_idx)-1 do begin
          col = pix_idx[k]+idx[0]
          pixfixrow,newimage,badmask,col,j,idx,_extra=e
        endfor

    endif
        
    endif
  endfor
endfor

RETURN,newimage
END
